require 'ansicolour'

class RepositoryFrontendTerm

    def initialize(args)
        @args = args
        @rep = Repository.new(args)
        @dists_comps = @rep.get_working_dists_comps(args[:dist], args[:component])
    end

    def add(locs)
        files = @rep.search_for_debs(locs)
        if files.empty?
            puts "ERROR: can't find any packages in '#{ARGV}'"
            exit(1)
        end
    
        _print_head
        files.keys.each do |file|
            puts file
        end
        print "\nAdd? (Y/n) "
        if @args[:dontask] == 1
            puts 'Y'
        elsif not [ 'a', 'y', 'Y', '' ].include?($stdin.gets.chomp)
            puts "Aborting"
            exit
        end
        _lock
        genpkgs = {}
        files.keys.each do |file|
            @dists_comps.keys.sort.each do |dist|
                @dists_comps[dist].keys.sort.each do |comp|
                    _putline("#{File.basename(file)} -> #{dist}/#{comp}")
                    pkg, msg = @rep.add_pkg(file, dist, comp)
                    _putstatus(pkg ? 'OK' : (msg ? msg : 'ERR'))
                    genpkgs[dist] or genpkgs[dist] = {}
                    genpkgs[dist][comp] = 1
                end
            end
        end
        puts
        _genpackages(genpkgs)
        _unlock
        @args[:nosync] == 1 or sync
    end


    def copy(pkgname, from, to)
        unless from
            puts "source dist/comp not set"
            exit(1)
        end
        unless to
            puts "target dist/comp not set"
            exit(1)
        end
        _lock
        sdist, scomp = from.split('/')
        tdist, tcomp = to.split('/')
        _putline("copying #{pkgname} #{sdist}/#{scomp} -> #{tdist}/#{tcomp}")
        pkg, msg = @rep.copy_pkg(pkgname, sdist, scomp, tdist, tcomp)
        _putstatus(pkg ? 'OK' : 'ERR', msg)
        _genpackages({sdist => {scomp => 1,}, tdist => {tcomp => 1,},})
        _unlock
        @args[:nosync] == 1 or sync
    end


    def move(pkgname, from, to)
        unless from
            puts "source dist/comp not set"
            exit(1)
        end
        unless to
            puts "target dist/comp not set"
            exit(1)
        end
        _lock
        sdist, scomp = from.split('/')
        tdist, tcomp = to.split('/')
        _putline("moving #{pkgname} #{sdist}/#{scomp} -> #{tdist}/#{tcomp}")
        pkg, msg = @rep.move_pkg(pkgname, sdist, scomp, tdist, tcomp)
        _putstatus(pkg ? 'OK' : 'ERR', msg)
        _genpackages({sdist => {scomp => 1,}, tdist => {tcomp => 1,},})
        _unlock
        @args[:nosync] == 1 or sync
    end


    def remove(patterns)
        _print_head
        list = patterns.map{|rexp| '^' + rexp + '$'}
        pkgs = @rep.search_pkgs(@dists_comps, list)
        if pkgs.empty?
            puts "ERROR: can't find any package matching any of '#{list.join(',')}' patterns"
            exit(1)
        end
        pkgs.keys.sort.each do |pkgname|
            print_pkg(pkgname, pkgs[pkgname], 'brief')
        end
        printf "\nDelete? (Y/n) "
        if @args[:dontask] == 1
            puts 'Y'
        elsif not [ 'a', 'y', 'Y', '' ].include?($stdin.gets.chomp)
            puts "Aborting"
            exit
        end
        _lock
        genpkgs = {}
        pkgs.keys.sort.each do |pkgname|
            pkgs[pkgname].keys.sort.each do |dist|
                pkgs[pkgname][dist].keys.sort.each do |comp|
                    _putline("removing #{pkgname} from #{dist}/#{comp}")
                    pkg, msg = @rep.remove_pkg(pkgname, dist, comp)
                    pkg ? _putstatus('OK') : _putstatus('ERR', msg)
                    genpkgs[dist] or genpkgs[dist] = {}
                    genpkgs[dist][comp] = 1
                end
            end
        end
        _genpackages(genpkgs)
        _unlock
        @args[:nosync] == 1 or sync
    end


    # vypise seznam baliku v danych dist/comp
    def list
        _print_head
        pkgs = @rep.get_pkgs(@dists_comps)
        pkgs.keys.sort.each do |pkgname|
            print_pkg(pkgname, pkgs[pkgname])
        end
    end


    def pkg_info(pkgname)
        _print_head
        pkgs = @rep.get_pkgs(@dists_comps)
        if pkgs.has_key?(pkgname)
            pkgs[pkgname].keys.sort.each do |dist|
                pkgs[pkgname][dist].keys.sort.each do |comp|
                    _print_pkg_info(pkgs[pkgname][dist][comp])
                end
            end
        else
            puts "ERROR: can't find any package '#{pkgname}'"
        end
    end


    def search(patterns)
        _print_head
        list = patterns.map{|rexp| '^' + rexp + '$'}
        pkgs = @rep.search_pkgs(@dists_comps, list)
        pkgs.keys.sort.each do |pkgname|
            print_pkg(pkgname, pkgs[pkgname], 'brief')
        end
    end


    def stats
        repo, dists_stats, dists = @rep.get_repository_stats(@dists_comps)
        printf "%-19s %-19s %-19s %-19s\n", 'dist/comp', 'num of pkgs', 'size in pool(MB)', 'installed size (MB)'
        puts
        @dists_comps.keys.sort.each do |dist|
            @dists_comps[dist].keys.sort.each do |comp|
                printf "%-19s %-19d %-19.2f %-19.2f\n", "#{dist}/#{comp}", dists[dist][comp]['count'], dists[dist][comp]['size']/1048576, dists[dist][comp]['isize']/1048576
            end
            printf "%-19s %-19d %-19.2f %-19.2f\n", "#{dist} (total)", dists_stats[dist]['count'], dists_stats[dist]['size']/1048576, dists_stats[dist]['isize']/1048576
            puts
        end
        printf "%-19s %-19d %-19.2f %-19.2f\n", "total", repo['count'], repo['size']/1048576, repo['isize']/1048576
    end


    def check_repository
        _lock
        _putline("checking files in pool (from DB package list)")
        check_repository_db2pool and _putstatus('OK')

        _putline("checking DB file list (from .deb files in pool)")
        check_repository_pool2db and _putstatus('OK')
        _unlock
    end


    def check_repository_db2pool
        @dists_comps = @rep.get_working_dists_comps([], [])
        genpkgs_db = {}
        pkgs = @rep.get_pkgs(@dists_comps)
        black_list = []
        pkgs.each do |pkgname, dists|
            dists.each do |distname, comps|
                comps.each do |compname, pkg|
                    file = "#{@args[:repository]}/#{pkg['Filename']}"
                    unless File.file?(file)
                        black_list.push("#{pkgname};#{distname};#{compname}")
                    end
                end
            end
        end
        if black_list.empty?
            return true
        end
        print "\n\n"
        puts "Following packages reference deb. files which don't exist in pool:"
        black_list.sort.each do |row|
            pkgname, dist, comp = row.split(';', 3)
            puts "#{pkgname} (#{dist}/#{comp})"
        end
        printf "\nDelete info about packages from DB? (Y/n) "
        if @args[:dontask] == 1
            puts 'Y'
        elsif not [ 'a', 'y', 'Y', '' ].include?($stdin.gets.chomp)
            puts "skipping"
            return false
        end
        rcode = true
        black_list.sort.each do |row|
            pkgname, dist, comp = row.split(';', 3)
            pkg, msg = @rep.remove_pkg(pkgname, dist, comp)
            unless pkg
                @log.error "err while removing #{pkg} from #{dist}/#{comp}: msg"
                rcode = false
            end
        end
        return rcode
    end

       

    def check_repository_pool2db
        reffiles = {}
        delfiles = {}
        @dists_comps = @rep.get_working_dists_comps([], [])
        pkgs = @rep.get_pkgs(@dists_comps)
        pkgs.each_value do |dist|
            dist.each_value do |comp|
                comp.each_value do |pkg|
                    reffiles[pkg['Filename']] = 1
                end
            end
        end
        pkgs = @rep.get_pkgs(@dists_comps)
        black_list = []
        p = IO.popen("find #{@args[:repository]}/pool/ -name '*deb'")
        p.each do |fname|
            fname.chomp!
            fname.sub!("#{@args[:repository]}/", '')
            unless reffiles[fname]
                black_list.push(fname)
            end
        end
        if black_list.empty?
            return true
        end
        print "\n\n"
        puts "Following file are not referenced by any package in DB:"
        black_list.sort.each do |fname|
            puts fname
        end
        printf "\nDelete this files? (Y/n) "
        if @args[:dontask] == 1
            puts 'Y'
        elsif not [ 'a', 'y', 'Y', '' ].include?($stdin.gets.chomp)
            puts "skipping"
            return false
        end
        rcode = true
        black_list.sort.each do |fname|
            begin
                File.unlink("#{@args[:repository]}/#{fname}")
            rescue
                @log.error "err while while deleting #{fname}: #{$!}"
                rcode = false
            end
        end
        return rcode
    end


    def getdists
        dists = @rep.parse_conf_file
        dists.keys.sort.each do |dist|
            puts dist
        end
    end


    def getcomps
        dists = @rep.parse_conf_file
        dist_list = @rep.args[:dist].empty? ? dists.keys : @rep.args[:dist]
        dist_list.sort.each do |dist|
            dists.has_key?(dist) or next
            dists[dist]['Components'] = [] unless dists[dist].has_key?('Components')
            puts dist + ': ' + dists[dist]['Components'].join(',')
        end
    end


    def sync
        #puts "synchronizaci jsem vypnul, abyste mi neloupali pernicek"
        #return

        
        printf "\nSync repository? (Y/n) "
        if @args[:dontask] == 1
            puts 'Y'
        elsif not [ 'a', 'y', 'Y', '' ].include?($stdin.gets.chomp)
            puts "Aborting"
            exit
        end
        _lock
        rcode, log = @rep.sync
        _putline("sync")
        if rcode == 0
            _putstatus('OK')
        else
            _putstatus('ERR', log.join(''))
        end
        _unlock
    end


    def genhtaccess
        require 'dealias'
        comp_access = _get_customers_for_comps($aliases)
        _lock
        @dists_comps.keys.sort.each do |dist|
            @dists_comps[dist].keys.sort.each do |comp|
                _putline("creating htaccess for #{dist}/#{comp}")
                rcode, log = @rep.gen_htaccess(comp_access, dist, comp)
                if rcode == 1
                    _putstatus('OK')
                else
                    _putstatus('ERR', log.join(''))
                end
            end
        end
        _unlock
        @args[:nosync] == 1 or sync
    end


    def genpackages
        dists = @rep.get_working_dists_comps([], [])
        _lock
        _genpackages(dists)
        _unlock
        @args[:nosync] == 1 or sync
    end

    def history(pkg)
        unless hist = @rep.get_pkg_history(pkg)
            puts "ERROR: there is no history for package '#{pkg}'"
            exit(1)
        end
        hist.each do |log|
            time = log.delete('time')
            user = log.delete('user')
            action = log.delete('action')
            args = ''
            log.keys.sort.each  do |k|
                args += "#{k}:#{log[k]};"
            end
            printf "%s (%s): %s -- %s\n", time, user, action, args 
        end
    end


    def newdist(dist)
        @rep.newdist(dist)
    end


    def newcomp(comp)
        @rep.newcomp(comp)
    end


    def unlock
        _putline("removing lock file")
        _putstatus(@rep.unlock ? 'OK' : 'ERR')
    end


    def print_pkg(pkgname, hash, type='brief')
        if @args[:print_format] == 'oneline'
            _print_pkg_oneline(pkgname, hash, type)
        else
            _print_pkg(pkgname, hash, type)
        end
    end

    def _print_pkg(pkgname, hash, type)
        puts pkgname
        hash.keys.sort.each do |dist|
            hash[dist].keys.sort.each do |comp|
                pkg = hash[dist][comp]
                printf("  %-18s %s\n", "#{dist}/#{comp}", pkg['Version'])
            end
        end
        puts
    end

    
    def _print_pkg_info(pkg)
        pkg.keys.sort.each do |var|
            next if var == 'LongDescription'
            print "#{var}: "
            puts pkg[var]
        end
        puts
    end


    def _lock
        unless @rep.lock
            puts "ERROR: failed to get lock, aborting"
            exit(1)
        end
    end


    def _unlock
        unless @rep.unlock
            puts "ERROR: failed to remove lock, aborting"
            exit(1)
        end
    end


    def _print_pkg_oneline(pkgname, hash, type)
        print "#{pkgname};"
        hash.keys.sort.each do |dist|
            hash[dist].keys.sort.each do |comp|
                pkg = hash[dist][comp]
                print "#{dist}/#{comp} #{pkg['Version']};"
            end
        end
        puts
    end


    # vygeneruje packages pro dists/comps
    def _genpackages(dists)
        dists.keys.sort.each do |dist|
            dists[dist].keys.sort.each do |comp|
                _putline("creating Packages for #{dist}/#{comp}")
                _putstatus(@rep.gen_packages_file(dist, comp) ? 'OK' : 'ERR')
            end
        end
    end


    def _get_customers_for_comps(aliases)
        comp_access = {}
        aliases.each_value do |customer|
            customer[:hosts].each do |host|
                next if host.has_key?(:no_maintain)
                pools = host.has_key?(:pools) ? host[:pools] : @args[:public_comps]
	            if !host.has_key?(:ip)
                    puts "#{host} has no ip, skipped"
                    next
                end
                ip = `gethostip -d #{host[:ip]} 2>/dev/null`.chomp
                if not ip or ip == ''
                    puts "can't resolve ip for #{host[:ip]}, skipped"
                    next
                end
                pools.each do |pool|
                    comp_access[pool] = {} unless comp_access.has_key?(pool)
                    comp_access[pool][ip] = host[:name]
                end
            end
        end
        return comp_access
    end


    def _print_head
        puts "ACTION:    : #{@args[:action]}"
        puts "COMPONENTS:"
        @dists_comps.keys.sort.each do |dist|
            print "  #{dist} ("
            print @dists_comps[dist].keys.sort.join(', ')
            puts ")"
        end
        print "\n\n"
    end


    def _putline(msg, type='term')
        #print msg
        #system("echo -n '#{msg} '")
        print msg
        length = 69-msg.length
        length = 3 if length < 3
        print '.' * length
        print ' '
        $stdout.flush
    end


    def _putstatus(msg, log=nil)
        if @args[:nocolours] == 1
            puts msg
        else
            if msg == 'OK'
                col = 'green'
            elsif msg == 'ERR'
                col = 'red'
            else
                col = 'white'
            end
            puts msg.colour(col,'bold')
        end
        $stdout.flush
        log and puts log
    end

end
