require 'repdb'
require 'fileutils'
require 'zlib'
require 'logger'

class Repository

    attr :args

    def initialize(args={})
        # get real user
        user = `id -un`.chomp
        user = ENV['SUDO_USER'] if ENV['SUDO_USER'] and user == 'debrp'

        @args = {
            :reprepro       => '',
            :verbose        => 0,
            :user           => user
        }.update(args)

	    @args[:logfile] = @args[:repository] + "/mainlog" unless @args.has_key?(':logfile')
	    @args[:lockfile] = @args[:repository] + "/lock" unless @args.has_key?(':lockfile')
	    @args[:pkglogdir] = @args[:repository] + "/logs" unless @args.has_key?(':logfile')
	    @args[:conf_file] = @args[:repository] + "/conf/distributions" unless @args.has_key?(':conf_file')

        @db = RepositoryDB.new({
            :pkgfile    => @args[:repository] + '/pkginfo/packages.db',
        })
        #@log = Logger.new(@args[:logfile])
        @log = Logger.new(STDOUT)
        @args[:loglevel] or @args[:loglevel] = 'error'
        case @args[:loglevel]
        when 'error'
            @log.level = Logger::ERROR
        when 'warn'
            @log.level = Logger::WARN
        when 'info'
            @log.level = Logger::INFO
        when 'debug'
            @log.level = Logger::DEBUG
        else
            @log.level = Logger::ERROR
        end
    end


    # prida deb. balik do dane distribuce/poolu
    # a do databaze baliku
    def add_pkg(file, dist, comp)
        @log.info "trying to add pkg #{file}"
        if not file or file == ''
            @log.error "filename is not set"
            return
        end
        if not dist or dist == ''
            @log.error "dist is not set"
            return
        end
        if not comp or comp == ''
            @log.error "component is not set"
            return
        end
        basefile = File.basename(file)
        # nacteme info z deb. baliku
        pkg = _extract_pkg_info(file)
        unless pkg['Package']
            @log.error "can't get package info from file '#{file}'"
            return
        end
        # urcime umisteni v poolu
        if pkg['Source']
            pool_path = _get_pool_path(pkg['Source'])
        else
            @log.debug "field Source is not set, using filename to resolve pool location"
            pool_path = _get_pool_path(file)
        end
        dst_dir = "#{@args[:repository]}/pool/#{comp}/#{pool_path}"
        pkg['_Component'] = comp
        pkg['_Distribution'] = dist
        pkg['Filename'] = "pool/#{comp}/#{pool_path}/#{basefile}"
        pkg['MD5sum'] = _get_md5sum(file)
        # pokud uz balik v dist/comp je, porovname, zda ma vyssi verzi
        if old_pkg = @db.get_pkg(pkg['Package'], dist, comp)
            @log.info "pkg '#{pkg['Package']} (#{pkg['Version']})' is already in #{dist}/#{comp}, comparing versions"
            if _is_newer_version?(pkg['Version'], old_pkg['Version']) or @args[:force] == 1
                # pokud ma existujici balik nizsi verzi, odstranime ho
                @log.info "existing package has version #{pkg['Version']} which is older than #{old_pkg['Version']}, replacing"
                unless remove_pkg(old_pkg['Package'], dist, comp)
                    @log.error "failed to remove older package, aborting"
                    return
                end
            else
                @log.info "existing package has version #{pkg['Version']} which is same or newer than #{old_pkg['Version']}, not adding"
                # FIXME
                return nil, "SKIPPED (already version #{old_pkg['Version']})"
            end
        end
        # z DB napred vezmeme seznam baliku se stejnou md5sumou, budeme
        # potrebovat pozdeji
        same_pkgs = @db.get_pkg_byname(pkg['Package'])
        # pridame balik do databaze
        @log.debug "adding pkg info to DB"
        @db.save_pkg(pkg)
        # zkopirujeme balik do repozitare
        unless File.directory?(dst_dir)
            @log.debug "creating dir '#{dst_dir}'"
            FileUtils.mkdir_p dst_dir
            #system("mkdir -p '#{dst_dir}'")
        end
        dst_file = "#{dst_dir}/#{basefile}"
        # pokud uz v poolu dany soubor existuje, tak ho napred smazem - nema tam
        # co delat
        File.unlink(dst_file) if File.file?(dst_file)
        # pokud uz v poolu je stejny balik, udelame hardlink
        created = false
        same_pkgs.each_value do |comps|
            @log.debug "searching deb pkg with same md5sum (#{pkg['MD5sum']})"
            comps.each_value do |same_pkg|
                next unless same_pkg['MD5sum'] == pkg['MD5sum']
                @log.debug "found pkg '#{same_pkg['Package']}' with same md5sum"
                unless File.file?("#{@args[:repository]}/#{same_pkg['Filename']}")
                    @log.debug "file '#{same_pkg['Filename']}' doesn't exist"
                    next
                end
                # soubory jsou stejne, nalinkujeme je
                if FileUtils.ln "#{@args[:repository]}/#{same_pkg['Filename']}", dst_file
                    @log.debug "created hardlink #{dst_file} on #{@args[:repository]}/#{same_pkg['Filename']}"
                    created = true
                    break
                end
            end
            created and break
            @log.debug "deb pkg with same md5sum not found"
        end
        if not created
            @log.debug "copying #{file} to #{dst_file}"
            FileUtils.copy(file, dst_file)
            File.chmod(0644, dst_file)
        end
        log_pkg_action(pkg['Package'], "action:add;dist:#{dist};comp:#{comp};name:#{file};version:#{pkg['Version']}")
        log_action("added package #{pkg['Filename']} to #{dist}/#{comp}")
        return pkg
    end


    # odstrani balik z dist/comp
    # vraci hash s infem odstraneneho baliku
    def remove_pkg(pkgname, dist, comp)
        @log.info "removing package #{pkgname} from #{dist}/#{comp}"
        unless pkg = @db.get_pkg(pkgname, dist, comp)
            @log.error "'#{pkgname}' not found in '#{dist}/#{comp}'"
            return
        end
        unless @db.remove_pkg(pkgname, dist, comp)
            @log.error "error while removing '#{pkgname}' from '#{dist}/#{comp}'"
            return
        end
        unless pkg['Filename'] or pkg['Filename'] == ''
            @log.error "pkg filename not set"
            return
        end
        unless pkg['Version'] or pkg['Version'] == ''
            @log.error "version is not set"
            return
        end
        # pokud uz tato verze v zadne distribuci neni, odstranime soubor z poolu
        delete_file = true
        same_pkgs = @db.get_pkg_byname(pkgname)
        same_pkgs.each do |comps|
            comps.each do |same_pkg|
                if same_pkg['Filename'] == pkg['Filename']
                    delete_file = false
                end
            end
        end
        if delete_file
            @log.info "removing pkgfile #{pkg['Filename']} from pool"
            filename = "#{@args[:repository]}/#{pkg['Filename']}"
            if File.file?(filename)
                File.unlink(filename)
            else
                @log.info "#{filename} doesn't exist"
            end
            # odstranime i adresare, pokud jsou prazdne
            # pool/multibox/z/zlib/zlib1g_1%3a1.2.3-6ubuntu4_i386.deb
            pool, comp, index, pkgname, file = pkg['Filename'].split('/')
            if pkgname and index
                begin
                    Dir.rmdir("#{@args[:repository]}/pool/#{comp}/#{index}/#{pkgname}/")
                    Dir.rmdir("#{@args[:repository]}/pool/#{comp}/#{index}/")
                rescue
                end
            end
        end
        log_pkg_action(pkg['Package'], "action:remove;dist:#{dist};comp:#{comp};version:#{pkg['Version']}")
        log_action("removed package #{pkg['Filename']} from #{dist}/#{comp}")
        return pkg
    end


    def get_pkgs_from_dist_comp(dist, comp)
        @log.info "getting pkgs from #{dist}/#{file}"
        return @db.get_pkgs_from_dist_comp(dist, comp)
    end


    def get_pkgs(dists)
        @log.info "getting pkgs from #{dists}"
        return @db.get_pkgs(dists)
    end


    # vraci hash baliku, jejichz jmeno splnuje 
    # dany reg. vyraz
    def search_pkgs(dists, regexps)
        @log.info "searching pkgs in #{dists} matching #{regexps}"
        pkgs = {}
        all_pkgs = @db.get_pkgs(dists)
        unless @args[:search_field]
            @log.info "searching pkgnames matching selected regexps"
            all_pkgs.keys.each do |pkgname|
                regexps.each do |re|
                    if pkgname.match(/#{re}/)
                        pkgs[pkgname] = all_pkgs[pkgname]
                    end
                end
            end
        else
            @log.info "searching pkgs with field #{@args[:search_field]} matching selected regexps"
            all_pkgs.each do |pkgname, dists|
                dists.each_value do |comps|
                    comps.each_value do |pkg|
                        pkg.has_key?(@args[:search_field]) or next
                        regexps.each do |re|
                            if pkg[@args[:search_field]].match(/#{re}/)
                                pkgs[pkgname] = all_pkgs[pkgname]
                            end
                        end
                    end
                end
            end
        end
        return pkgs
    end


    def check_repository
        puts "jojo, pohodicka"
    end


    # vraci distribuce a komponenty nad kterymi budeme pracovat
    def get_working_dists_comps(dists=[], comps=[])
        hash = {}
        default = parse_conf_file
        # pokud nebyla specifikovana zadna distribuce, pouzijeme vsechny
        dists = default.keys if dists.empty?
        dists.each do |dist|
            # pokud takovy distro neexistuje, preskocime ho
            unless default.has_key?(dist)
                @log.info "'#{dist}' is not in list of dists, skipping"
                next
            end
            hash[dist] = {}
            dcomps = comps.empty? ? default[dist]['Components'] : comps
            dcomps.each do |comp|
                # pokud takova komponenta neexistuje, preskocime ji
                default[dist]['Components'].index(comp) or next
                hash[dist][comp] = 1
            end
        end
        if hash.empty?
            @log.error "list of valid dists/comps is empty"
            exit(1)
        end
        return hash
    end


    # vraci tri hashe - se sumarnim info o repozitari, se sumarnimi 
    # informacemi o distribucich a s informacemi o dist/comp
    def get_repository_stats(dists_tree)
        dists = {}
        repo = {}
        repo['size'] = repo['isize'] = 0.to_f
        repo['count'] = 0
        dists_stats = {}
        dists_tree.keys.sort.each do |dist|
            dists[dist] = {}
            dists_stats[dist] = {}
            dists_stats[dist]['size'] =  0.to_f
            dists_stats[dist]['count'] =  0
            dists_stats[dist]['isize'] =  0.to_f
            dists_tree[dist].keys.sort.each do |comp|
                count = size = isize = 0
                pkgs = @db.get_pkgs_from_dist_comp(dist, comp)
                pkgs.each do |pkgname, pkg|
                    count += 1
                    isize += pkg['Installed-Size'].to_f * 1024
                    file = "#{@args[:repository]}/#{pkg['Filename']}"
                    if File.file?(file)
                        size += File.size(file)
                    end
                end
                dists[dist][comp] = {}
                dists[dist][comp]['size'] = size
                dists[dist][comp]['isize'] = isize
                dists[dist][comp]['count'] = count
                dists_stats[dist]['size'] +=  size
                dists_stats[dist]['count'] +=  count
                dists_stats[dist]['isize'] +=  isize
            end
            repo['count'] += dists_stats[dist]['count']
            repo['size'] += dists_stats[dist]['size']
            repo['isize'] += dists_stats[dist]['isize']
        end
        return repo, dists_stats, dists
    end


    # vygeneruje soubor Packages v dists/... podadresari
    def gen_packages_file(dist, comp)
        fields = %w(Package Priority Section Installed-Size Maintainer 
        Architecture Source Version Replaces Provides Depends Predepends 
        Recommends Suggests Conflicts Filename Size MD5sum Description)
        pkg_dir = "#{@args[:repository]}/dists/#{dist}/#{comp}/binary-i386"
        File.directory?(pkg_dir) or FileUtils.mkdir_p pkg_dir
        fd = File.new("#{pkg_dir}/Packages", 'w', 0644)
        fdz = Zlib::GzipWriter.open("#{pkg_dir}/Packages.gz")
        pkgs = @db.get_pkgs_from_dist_comp(dist, comp)
        pkgs.keys.sort.each do |pkgname|
            rec = ''
            fields.each do |f|
                pkgs[pkgname].has_key?(f) or next
                rec += "#{f}: #{pkgs[pkgname][f]}\n"
                pkgs[pkgname].delete(f)
            end
            rec += "#{pkgs[pkgname]['LongDescription']}\n"
            pkgs[pkgname].delete('LongDescription')
            # vypisem i zbyvajici hodnoty:
            pkgs[pkgname].each do |var, val|
                var.match(/^_/) and next
                rec += "#{var}: #{val}\n"
            end
            rec += "\n"
            fd.puts rec
            fdz.puts rec
        end
        fdz.close
        @log.info "generated #{pkg_dir}/Packages"
        @log.info "generated #{pkg_dir}/Packages.gz"
        log_action("generated #{pkg_dir}/Packages(.gz)")
        return true
    end


    # syncne dist a pool s hostingem 
    def sync
        default_dists = parse_conf_file 
        dist = default_dists.keys
        # zkusime syncovat celej pool - je to pohodlnejsi ;-) (a hlavne bezpecnejsi - hardlinky)
        log_action("SYNC: syncing repository")
        #rcode, log = _run_rsync("rsync --delete --progress -e 'ssh -i /home/solnet/root_key/hosting_www_debrp_dsa' -aztH #{@args[:repository]}/pool/ #{@args[:sync_host]}:#{@args[:sync_dir]}/pool/ 2>&1")
        rcode, log = _run_rsync("rsync --delete --progress -e 'ssh -i /home/solnet/root_key/hosting_www_debrp_dsa -o StrictHostKeyChecking=no' -aztH #{@args[:repository]}/dists #{@args[:repository]}/pool  #{@args[:sync_host]}:#{@args[:sync_dir]}/ 2>&1", 1)
        return rcode, log
    end


    # vygeneruje soubory .htaccess v zavislosti na aliases pro pool/ a dists/
    def gen_htaccess(comp_access, dist, comp)
        default_template = "#{@args[:repository]}/conf/htaccess_templates/htaccess_tamplate"
        generated_comps = {}
        log_action("generating htaccess for dist/#{dist}/#{comp}")
        template = "#{default_template}_#{comp}"
        template = default_template unless File.file?(template)
        @log.info "reading htaccess template from #{template}"
        lines = File.readlines(template)
        new_lines = []
        lines.each do |line|
            unless line.match(/^#SOL_DPKG\s+(\w+)=([\w,]+)$/)
                new_lines.push(line)
                next
            end
            if $1 == 'components'
                comps_allowed = $2.split(',')
                if comps_allowed.empty?
                    puts "components not specified"
                    next
                end
                ip_list = {}
                comps_allowed.each do |c|
                    ip_list.merge!(comp_access[c]) if comp_access.has_key?(c)
                end
                ip_list.each do |ip, host|
                    new_lines.push("# #{host}\n")
                    new_lines.push("Allow from #{ip}\n")
                end
            else
                puts "parse error - unknown variable '#{$1}'"
            end
        end
        htaccess_dist_dir = "#{@args[:repository]}/dists/#{dist}/#{comp}"
        htaccess_comp_dir = "#{@args[:repository]}/pool/#{comp}"
        Dir.mkdir(htaccess_dist_dir) unless File.directory?(htaccess_dist_dir)
        File.open("#{htaccess_dist_dir}/.htaccess", 'w') { |f|
            new_lines.each { |line| f.puts(line) }
        }
        log_action("generated #{htaccess_dist_dir}/.htaccess")
        Dir.mkdir(htaccess_comp_dir) unless File.directory?(htaccess_comp_dir)
        File.open("#{htaccess_comp_dir}/.htaccess", 'w') { |f|
            new_lines.each { |line| f.puts(line) }
        }
        log_action("generated #{htaccess_comp_dir}/.htaccess")
        return 1
    end


    # pro zadane cesty (abs./rel. cesty k adersarum/souborum)
    # vraci seznam nalezenych deb. baliku
    def search_for_debs(paths)
        def parse_files_file(dir)
            file = "#{dir}/debian/files"
            files = []
            debdir = dir.sub(/\/[^\/]+\/?$/,'')
            unless File.file?(file)
                @log.error "not debian source dir ('#{file}' doesn't exist)"
                return files
            end
            open(file,'r') do |f|
                f.each do |line|
                    line.chomp!
                    line.sub!(/ .*/,'')
                    files.push(line)
                end
            end
            # pri nenastavene distribuci ji nastavime:
            if @args[:dist].empty? and not files.empty?
                ver = files[0].sub(/.*_([^_]+)_.*/, '\1')
                # nas balik
                if ver.sub!(/^(2\d\d\d).(Q\d).\d+$/, '\1\2')
                    @args[:dist] = [ ver ]
                end
            end
            return files
        end

        def detect_debs(path)
            def resolve_file_type(path)
                #if path.match(/.*\.changes$/)
                #    @log.detected '#{path}' .changes file"
                #    return 'include'
                #elsif path.match(/.*\.deb$/)
                if path.match(/.*\.deb$/)
                    #@log.detected '#{path}' .deb file"
                    return 'includedeb'
                end
                return ''
            end
            files = {}
            path.match(/^\//) or path = "#{Dir.pwd}/#{path}"
            if File.file?(path)
                type = resolve_file_type(path)
                type.empty? or files[path] = type
            elsif File.directory?(path)
                dir = Dir.new(path)
                dir.each do |file|
                    file = "#{path}/#{file}"        
                    File.file?(file) or next
                    type = resolve_file_type(file)
                    type.empty? or files[file] = type
                end
            else
                puts "WARN: path '#{path} doesn't exists, skipping"
            end
            return files
        end

        files = {}
        if paths.length == 0
            # pokud jsme v nadrazenym branch adreasri, musime urcit adresar s balikem
            pkgname = `head -1 solnet/PKGNAME 2>/dev/null`.chomp
            srcdir = Dir.pwd
            debdir = srcdir.sub(/\/[^\/]+\/?$/, '')
            if pkgname != ''
                @log.info "solnet branch dir, reading from solnet/PKGNAME..."
                subdir = `find . -maxdepth 1 -name '#{pkgname}*'|head -1 2>/dev/null`.chomp
                srcdir = Dir.pwd+"/#{subdir}"
                debdir = Dir.pwd+"/build"
            end
            paths = []
            parse_files_file(srcdir).each do |p|
                paths.push("#{debdir}/#{p}")
            end
        end
        paths.each do |path|
            files.merge!(detect_debs(path))
        end
        return files
    end


    # urci adresar pro balik souboru v poolu, vraci String
    # napr. pro balik 3ddesktop vrati '3/3ddesktop'
    # pro /tmp/lib-asdf_213.deb vrazti 'lib-/lib-asdf'
    def _get_pool_path(filename)
        filename = File.basename(filename)
        filename.sub!(/_.*/, '')
        index = (filename.match(/^lib/) ? filename[0..3] : filename[0..0])
        return "#{index}/#{filename}"
    end


    # ze zadaneho deb. baliku precte info (podobne jako vystup z dpkg --info
    # vraci hash, kde timto info
    def _extract_pkg_info(file)
        info = {}
        info['LongDescription'] = ''
        cmd = "ar p '#{file}' control.tar.gz |tar -O -xz ./control 2>/dev/null"
        pipe = IO.popen(cmd)
        pipe.each do |line|
            if line.match(/^ /)
                info['LongDescription'] += line
                next
            end
            line.chomp!
            unless line.match(/:\s*/)
                @log.debug "line '#{line}' doesn't contain ':', skipping"
                next
            end
            val, var = line.split(/:\s*/, 2)
            if val.match(/'/)
                @log.debug "skipping line #{line} - field name has illegal chars"
                next
            end
            info[val] = var
        end
        pipe.close
        return info
    end


    # porovnava debianni verze 
    def _is_newer_version?(new, old)
        cmd = "dpkg --compare-versions '#{new}' 'gt' '#{old}' 2>/dev/null"
        return system(cmd)
    end


    # vraci md5 sumu zadaneho souboru
    def _get_md5sum(file)
        sum = `/usr/bin/md5sum '#{file}'`
        sum.chomp!
        sum.sub!(/\s.*/, '')
        return sum
    end


    def _run_rsync(cmd, mode=0)
        log = []
        pipe = IO.popen(cmd)
        pipe.each do |line|
            log.push(line)
            mode == 0 and next
            line.match(/\/$/) and mode == 1 and next
            line.match(/^\s/) and mode == 1 and next
            puts line
        end
        pipe.close
        return $?, log
    end

 
    def get_pkg_history(pkgname)
        file = "#{@args[:pkglogdir]}/#{pkgname}"
        return unless File.file?(file)
        log = []
        open(file,'r') do |f|
            f.each do |line|
                line.chomp!
                items = line.split(';')
                hash = {}
                items.each do |i|
                    var, val = i.split(':')
                    var == nil and next
                    hash[var] = val
                end
                log.push(hash)
            end
        end
        return log
    end


    def newdist(dist)
        if dist == nil or not dist.match(/^\w+$/)
            puts "ERR: '#{dist}' is not correct dist name, aborting"
            return
        end
        cfile = parse_conf_file 
        dists = cfile.keys;
        if dists.index(dist)
            puts "ERR: '#{dist}' already exists, aborting"
            return
        end
        open(@args[:conf_file],'a') do |f|
            f.print <<EOF

Origin: soLNet
Label: #{dist}
Suite: stable
Codename: #{dist}
Version: #{dist}
Architectures: i386
Components: adopted open multibox filebox firewall mailbox backports
Description: soLNet repository
EOF
        end
        puts "OK, new distro added to '#{@args[:conf_file]}', check manually"
        log_action("created new distribution '#{dist}'")
        return 1
    end
   

    def newcomp(arg)
        if arg == nil
            puts "ERR: argument is nil, aborting"
            return
        end
        dist, comp = arg.split('/')
        cfile = parse_conf_file 
        dists = cfile.keys;
        if not comp.match(/^[\w\-\.]+$/)
            puts "ERR: '#{comp}' is not correct compname"
            return
        end
        if not dists.index(dist)
            puts "ERR: dist '#{dist}' doesn't exist, aborting"
            return
        end
        if cfile[dist]['Components'] and cfile[dist]['Components'].index(comp)
            puts "ERR: comp '#{comp}' already exists in dist '#{dist}'"
            return
        end
        # FIXME: hack :-/
        lines = IO.readlines(@args[:conf_file])
        state = changed = 0
        lines.each_index do |i|
            if lines[i].match(/^Label:\s*#{dist}\s*$/)
                state = 1
                next
            end
            if lines[i].match(/^Components:/) and state == 1
                lines[i].chomp!
                lines[i] = "#{lines[i]} #{comp}\n"
                changed = 1
                break
            end
        end
        if changed == 0
            puts "WARN: file not modified, check Components line (#{@args[:conf_file]})"
            return
        end
        f = File.open("#{@args[:conf_file]}.tmp", 'w')
        f << lines
        f.close
        File.rename("#{@args[:conf_file]}.tmp", @args[:conf_file])
        log_action("created new component '#{dist}/#{comp}'")
        puts "OK, created new component '#{dist}/#{comp}'"
        return 1
    end

 
    def parse_conf_file
        opts = {}
        dists = {}
        unless File.file?(@args[:conf_file])
            puts "can't read config file '#{@args[:conf_file]}'"
            return
        end
        open(@args[:conf_file],'r') do |f|
            f.each do |line|
                line.chomp!
                if line.match(/^\s*$/) and opts.has_key?('Codename')
                    dists[opts['Codename']] = opts
                    opts = {}
                elsif md = line.match(/:\s*/)
                    if md.pre_match == 'Components'
                        comps = md.post_match.split(/\s+/)
                        opts[md.pre_match] = comps
                    else
                        opts[md.pre_match] = md.post_match
                    end
                end
            end
            if opts.has_key?('Codename')
                dists[opts['Codename']] = opts
            end
        end
        return dists
    end


    def copy_pkg(pkgname, sdist, scomp, tdist, tcomp)
        # pokud zdrojovy balik neexistuje, tak koncime
        unless pkg = @db.get_pkg(pkgname, sdist, scomp)
            return nil, "pkg '#{pkgname}' is not in #{sdist}/#{scomp}"
        end
        # pokud uz tento balik v cilove distribuci existuje, tak
        # koncime
        if @db.get_pkg(pkgname, tdist, tcomp)
            return nil, "pkg '#{pkgname}' is already in #{tdist}/#{tcomp}"
        end
        sfile = "#{@args[:repository]}/#{pkg['Filename']}"
        unless File.file?(sfile)
            return nil, "file '#{sfile}' doesn't exist"
        end
        return add_pkg(sfile, tdist, tcomp)
    end


    def move_pkg(pkgname, sdist, scomp, tdist, tcomp)
        moved_pkg, msg = copy_pkg(pkgname, sdist, scomp, tdist, tcomp)
        unless moved_pkg
            return nil, msg
        end
        return remove_pkg(pkgname, sdist, scomp)
    end

    
    def locked_by
        return unless File.file?(@args[:lockfile])
        @log.debug "trying to read lockfile '#{@args[:lockfile]}'"
        begin
            user = File.readlines(@args[:lockfile])[0]
            return user ? user : ''
        rescue
            @log.warn "failed to read lockfile '#{@args[:lockfile]}': #{$!}"
            return ''
        end
    end


    def lock
        if user = locked_by
            @log.error "lockfile '#{@args[:lockfile]}' already exists (user '#{user}')"
            return false
        end
        @log.debug "trying to create lockfile '#{@args[:lockfile]}'"
        begin
            f = File.open(@args[:lockfile], 'w')
            f.print @args[:user]
            f.close
        rescue
            @log.error "failed to create lockfile '#{@args[:lockfile]}': #{$!}"
            return false
        end
        return true
    end


    def unlock
        unless File.file?(@args[:lockfile])
            @log.error "lockfile '#{@args[:lockfile]}' doesn't exists"
            return false
        end
        @log.debug "trying to remove lockfile '#{@args[:lockfile]}'"
        begin
            File.unlink(@args[:lockfile])
        rescue
            @log.error "failed to remove lockfile '#{@args[:lockfile]}': #{$!}"
            return false
        end
        return true
    end


#    def get_pkgs_deps(dist)
#        dist, comp, pkg_re = dist.split('/')
#        dists = parse_conf_file
#        unless dists.has_key?(dist)
#            puts "ERROR: can't find distribution '#{dist}'"
#            exit(1)
#        end
#        comps = (comp == nil) ? dists[dist]['Components'] : comp.split(',')
#        deps = {}
#        pkgs = get_pkgs_info([dist], comps)
#        # FIXME: showdeps je volano na _jednu_ distribuci - muzem predpokladat,
#        # ze pokud je balik ve vice komponentach, je ve vsech stejna verze
#        pkgs.each do |pname, pkg_array|
#            pkg_array.each do |pkg|
#                next if pkg_re and not pkg_re.empty? and not pname.match(/#{pkg_re}/)
#                deps[pname] = {
#                    'out'           => [],
#                    'comp'          => [],
#                    'comp-same'     => [],
#                    'out-version'   => [],
#                }
#                next unless pkg.has_key?('Depends')
#                pkg['Depends'].keys.each do |dep|
#                    if not pkgs.has_key?(dep)
#                        # pokud neni v seznamu vsech baliku, je to zavislost na cizim baliku
#                        deps[pname]['out'].push(dep)
#                    elsif not pkgs[dep][0].has_key?('Version')
#                        # pokud je v seznamu, ale nema verzi, tak ho ignorujem
#                        deps[pname]['out'].push(dep)
#                    elsif (pkg['Depends'][dep] == '' or comp_versions(pkgs[dep][0]['Version'], pkgs[pname][0]['Version']) == 1)
#                        # pokud znezavisi na konkretni verzi, nebo je verze ok
#                        comps_dep = pkgs[dep].map { |p| p['comp'] }
#                        comps_pkg = pkgs[pname].map { |p| p['comp'] }
#                        intsec = comps_dep & comps_pkg
#                        if intsec.empty?
#                            deps[pname]['comp'].push(dep)
#                        else
#                            deps[pname]['comp-same'].push(dep)
#                        end
#                    else
#                        deps[pname]['out-version'].push(dep)
#                    end
#                end
#            end
#        end
#        return deps
#    end


#    def parse_packages_file(file)
#        f = File.new(file, "r")
#        pkgs = {}
#        opts = {}
#        f.each_line do |line|
#            line.chomp!
#            if line.match(/^$/) and opts.has_key?('Package')
#                if opts.has_key?('Depends')
#                    deps = {}
#                    opts['Depends'].split(/,\s*/).each do |p|
#                        if p.match(/^([^\s]+)\s+\((.*)\)$/)
#                            deps[$1] = $2
#                        else
#                            deps[p] = ''
#                        end
#                    end
#                    opts['Depends'] = deps
#                end
#                pkgs[opts['Package']] = opts
#                opts = {}
#            elsif md = line.match(/^([\w-]+): /)
#                opts[$1] = md.post_match
#            end
#        end
#        if opts.has_key?('Package')
#            pkgs[opts['Package']] = opts
#        end
#        f.close
#        return pkgs
#    end


   
    ##### "interni" fce #####


    def _log(level, msg)
        #open(@args[:logfile],'a') do |f|
        #    f.print Time.now.strftime("%Y.%m.%d %H:%M:%S"), "(#{@args[:user]}): "
        #    f.puts msg
        #end
        #if @args[:verbose] >= level
        #    puts msg
        #end
    end
    

    def log_action(msg)
        open(@args[:logfile],'a') do |f|
            f.print Time.now.strftime("%Y.%m.%d %H:%M:%S"), "(#{@args[:user]}): "
            f.puts msg
        end
    end


    def log_pkg_action(pkg, msg)
        open("#{@args[:pkglogdir]}/#{pkg}",'a') do |f|
            f.print Time.now.strftime("time:%Y.%m.%d %H:%M:%S"), ';', "user:#{@args[:user]};"
            f.puts msg
        end
    end

end

